/* =========================
   Premium Portfolio — JS
   - Theme toggle (localStorage)
   - Mobile nav
   - Scroll spy
   - Reveal on view (IntersectionObserver)
   - Projects filter
   ========================= */

(function () {
  const root = document.documentElement;

  // ---------- Theme ----------
  const THEME_KEY = "gc_portfolio_theme";
  const themeBtn = document.getElementById("themeBtn");

  function getPreferredTheme() {
    const saved = localStorage.getItem(THEME_KEY);
    if (saved === "dark" || saved === "light") return saved;

    // Default: dark (premium look). You can switch to prefers-color-scheme if you prefer.
    return "dark";
  }

  function applyTheme(theme) {
    if (theme === "light") {
      root.setAttribute("data-theme", "light");
    } else {
      root.removeAttribute("data-theme"); // dark default via :root
    }
    localStorage.setItem(THEME_KEY, theme);
    updateThemeBtnLabel(theme);
  }

  function updateThemeBtnLabel(theme) {
    if (!themeBtn) return;
    const text = themeBtn.querySelector(".themeBtn__text");
    if (text) text.textContent = theme === "light" ? "Light" : "Dark";
    themeBtn.setAttribute("aria-label", `Basculer le thème (actuel : ${theme})`);
  }

  if (themeBtn) {
    applyTheme(getPreferredTheme());
    themeBtn.addEventListener("click", () => {
      const current = localStorage.getItem(THEME_KEY) || getPreferredTheme();
      const next = current === "light" ? "dark" : "light";
      applyTheme(next);
    });
  }

  // ---------- Footer year ----------
  const yearEl = document.getElementById("year");
  if (yearEl) yearEl.textContent = new Date().getFullYear();

  // ---------- Mobile nav ----------
  const navToggle = document.querySelector(".nav__toggle");
  const navMenu = document.getElementById("navMenu");

  function setMenu(open) {
    if (!navMenu || !navToggle) return;
    navMenu.classList.toggle("is-open", open);
    navToggle.setAttribute("aria-expanded", String(open));
  }

  if (navToggle && navMenu) {
    navToggle.addEventListener("click", () => {
      const isOpen = navMenu.classList.contains("is-open");
      setMenu(!isOpen);
    });

    // Close on link click
    navMenu.addEventListener("click", (e) => {
      const a = e.target.closest("a");
      if (a) setMenu(false);
    });

    // Close on outside click
    document.addEventListener("click", (e) => {
      if (!navMenu.classList.contains("is-open")) return;
      const clickedInside = e.target.closest(".nav") || e.target.closest(".nav__toggle");
      if (!clickedInside) setMenu(false);
    });

    // Close on ESC
    document.addEventListener("keydown", (e) => {
      if (e.key === "Escape") setMenu(false);
    });
  }

  // ---------- Reveal on view ----------
  const reveals = Array.from(document.querySelectorAll(".reveal"));
  const io = new IntersectionObserver(
    (entries) => {
      entries.forEach((entry) => {
        if (entry.isIntersecting) {
          entry.target.classList.add("is-in");
          io.unobserve(entry.target);
        }
      });
    },
    { threshold: 0.12 }
  );
  reveals.forEach((el) => io.observe(el));

  // ---------- Projects filter ----------
  const filterBtns = Array.from(document.querySelectorAll(".filterBtn"));
  const projectCards = Array.from(document.querySelectorAll(".projCard"));
  const projectsGrid = document.getElementById("projectsGrid");

  function setAriaTabs(activeBtn) {
    filterBtns.forEach((btn) => {
      const isActive = btn === activeBtn;
      btn.classList.toggle("is-active", isActive);
      btn.setAttribute("aria-selected", String(isActive));
    });
  }

  function filterProjects(filter) {
    // Make grid stable when hiding elements: temporarily set min-height
    if (projectsGrid) {
      const rect = projectsGrid.getBoundingClientRect();
      projectsGrid.style.minHeight = `${Math.max(rect.height, 120)}px`;
      setTimeout(() => (projectsGrid.style.minHeight = ""), 250);
    }

    projectCards.forEach((card) => {
      const cats = (card.getAttribute("data-category") || "").split(/\s+/).filter(Boolean);
      const shouldShow = filter === "all" || cats.includes(filter);

      // Simple animation: hide via class
      if (shouldShow) {
        card.classList.remove("is-hidden");
        card.style.position = "";
      } else {
        card.classList.add("is-hidden");
        card.style.position = "absolute";
      }
    });
  }

  if (filterBtns.length && projectCards.length) {
    filterBtns.forEach((btn) => {
      btn.addEventListener("click", () => {
        const filter = btn.getAttribute("data-filter") || "all";
        setAriaTabs(btn);
        filterProjects(filter);
      });
    });
  }

  // ---------- Scroll spy (active link) ----------
  const navLinks = Array.from(document.querySelectorAll(".nav__link"));
  const sectionIds = navLinks
    .map((a) => a.getAttribute("href"))
    .filter((h) => h && h.startsWith("#"))
    .map((h) => h.slice(1));

  const sections = sectionIds
    .map((id) => document.getElementById(id))
    .filter(Boolean);

  function setActiveLink(id) {
    navLinks.forEach((a) => {
      const href = a.getAttribute("href");
      const active = href === `#${id}`;
      a.classList.toggle("is-active", active);
    });
  }

  const spy = new IntersectionObserver(
    (entries) => {
      // pick the most visible
      const visible = entries
        .filter((e) => e.isIntersecting)
        .sort((a, b) => b.intersectionRatio - a.intersectionRatio)[0];

      if (visible && visible.target && visible.target.id) {
        setActiveLink(visible.target.id);
      }
    },
    { rootMargin: "-15% 0px -70% 0px", threshold: [0.1, 0.2, 0.35, 0.5, 0.65] }
  );

  sections.forEach((s) => spy.observe(s));
})();
